/**
 * Copyright (c) 2012 Jean-Christophe Dubois.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @file brd_main.c
 * @author Jean-Christophe Dubois (jcd@tribudubois.net)
 * @brief main source file for board specific code
 */

#include <vmm_error.h>
#include <vmm_string.h>
#include <vmm_devtree.h>
#include <vmm_devdrv.h>
#include <vmm_host_io.h>
#include <vmm_host_aspace.h>
#include <vmm_stdio.h>
#include <vmm_chardev.h>
#include <rtc/vmm_rtcdev.h>
#include <libfdt.h>
#include <versatile_plat.h>
#include <versatile_board.h>
#include <sp804_timer.h>

extern u32 dt_blob_start;
static virtual_addr_t versatile_sys_base;

int arch_board_ram_start(physical_addr_t * addr)
{
	struct fdt_fileinfo fdt;
	struct fdt_node_header *fdt_node;
	struct fdt_property *prop;

	*addr = 0xffffffff;

	if (libfdt_parse_fileinfo((virtual_addr_t) & dt_blob_start, &fdt)) {
		return VMM_EFAIL;
	}

	fdt_node = libfdt_find_node(&fdt,
				    VMM_DEVTREE_PATH_SEPARATOR_STRING
				    VMM_DEVTREE_HOSTINFO_NODE_NAME
				    VMM_DEVTREE_PATH_SEPARATOR_STRING
				    VMM_DEVTREE_MEMORY_NODE_NAME);
	if (!fdt_node) {
		return VMM_EFAIL;
	}

	prop = libfdt_get_property(&fdt, fdt_node,
				   VMM_DEVTREE_MEMORY_PHYS_ADDR_ATTR_NAME);

	if (!prop) {
		return VMM_EFAIL;
	}

	*addr = *((physical_addr_t *) prop->data);

	return VMM_OK;
}

int arch_board_ram_size(physical_size_t * size)
{
	struct fdt_fileinfo fdt;
	struct fdt_node_header *fdt_node;
	struct fdt_property *prop;

	*size = 0;

	if (libfdt_parse_fileinfo((virtual_addr_t) & dt_blob_start, &fdt)) {
		return VMM_EFAIL;
	}

	fdt_node = libfdt_find_node(&fdt,
				    VMM_DEVTREE_PATH_SEPARATOR_STRING
				    VMM_DEVTREE_HOSTINFO_NODE_NAME
				    VMM_DEVTREE_PATH_SEPARATOR_STRING
				    VMM_DEVTREE_MEMORY_NODE_NAME);
	if (!fdt_node) {
		return VMM_EFAIL;
	}

	prop = libfdt_get_property(&fdt, fdt_node,
				   VMM_DEVTREE_MEMORY_PHYS_SIZE_ATTR_NAME);
	if (!prop) {
		return VMM_EFAIL;
	}

	*size = *((physical_size_t *) prop->data);

	return VMM_OK;
}

int arch_board_devtree_populate(struct vmm_devtree_node **root)
{
	struct fdt_fileinfo fdt;

	if (libfdt_parse_fileinfo((virtual_addr_t) & dt_blob_start, &fdt)) {
		return VMM_EFAIL;
	}

	return libfdt_parse_devtree(&fdt, root);
}

int arch_board_reset(void)
{
	vmm_writel(0x101,
		   (void *)(versatile_sys_base +
			    VERSATILE_SYS_RESETCTL_OFFSET));

	return VMM_OK;
}

int arch_board_shutdown(void)
{
	/* FIXME: TBD */
	return VMM_OK;
}

int __init arch_board_early_init(void)
{
	/*
	 * TODO:
	 * Host virtual memory, device tree, heap is up.
	 * Do necessary early stuff like iomapping devices
	 * memory or boot time memory reservation here.
	 */
	return VMM_OK;
}

int __init arch_board_final_init(void)
{
	struct vmm_devtree_node *node;
	struct vmm_chardev *cdev;
#if defined(CONFIG_RTC)
	struct vmm_rtcdev * rdev;
#endif

	/* All VMM API's are available here */
	/* We can register a Board specific resource here */

	/* Map control registers */
	versatile_sys_base = vmm_host_iomap(VERSATILE_SYS_BASE, 0x1000);

	/* Unlock Lockable registers */
	vmm_writel(VERSATILE_SYS_LOCKVAL,
		   (void *)(versatile_sys_base + VERSATILE_SYS_LOCK_OFFSET));

	/* Do Probing using device driver framework */
	node = vmm_devtree_getnode(VMM_DEVTREE_PATH_SEPARATOR_STRING
				   VMM_DEVTREE_HOSTINFO_NODE_NAME
				   VMM_DEVTREE_PATH_SEPARATOR_STRING "nbridge");

	if (!node) {
		return VMM_ENOTAVAIL;
	}

	if (vmm_devdrv_probe(node, NULL)) {
		return VMM_EFAIL;
	}

	/* Find uart0 character device and 
	 * set it as vmm_stdio character device */
	if ((cdev = vmm_chardev_find("uart0"))) {
		vmm_stdio_change_indevice(cdev);
		vmm_stdio_change_outdevice(cdev);
	}

	/* Syncup wall-clock time from rtc0 */
#if defined(CONFIG_RTC)
	if ((rdev = vmm_rtcdev_find("rtc0"))) {
		int rc;
		if ((rc = vmm_rtcdev_sync_wallclock(rdev))) {
			return rc;
		}
	}
#endif
	return VMM_OK;
}
