/**
 * Copyright (c) 2015 Anup Patel.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @file zero.c
 * @author Anup Patel (anup@brainfault.org)
 * @brief Simple pass-through emulator.
 *
 * This emulator to should be use for pass-through access to non-DMA
 * capable device which do not require IOMMU, CLK, and PINMUX configuration.
 */

#include <vmm_error.h>
#include <vmm_heap.h>
#include <vmm_stdio.h>
#include <vmm_host_irq.h>
#include <vmm_modules.h>
#include <vmm_devtree.h>
#include <vmm_devemu.h>

#define MODULE_DESC			"Simple Pass-through Emulator"
#define MODULE_AUTHOR			"Anup Patel"
#define MODULE_LICENSE			"GPL"
#define MODULE_IPRIORITY		0
#define	MODULE_INIT			simple_emulator_init
#define	MODULE_EXIT			simple_emulator_exit

struct simple_state {
	char name[64];
	struct vmm_guest *guest;
	u32 irq_count;
	u32 *host_irqs;
	u32 *guest_irqs;
};

/* Handle host-to-guest routed IRQ generated by device */
static vmm_irq_return_t simple_routed_irq(int irq, void *dev)
{
	int rc;
	bool found = FALSE;
	u32 i, host_irq = irq, guest_irq = 0;
	struct simple_state *s = dev;

	/* Find guest irq */
	for (i = 0; i < s->irq_count; i++) {
		if (s->host_irqs[i] == host_irq) {
			guest_irq = s->guest_irqs[i];
			found = TRUE;
			break;
		}
	}
	if (!found) {
		goto done;
	}

	/* Lower the interrupt level.
	 * This will clear previous interrupt state.
	 */
	rc = vmm_devemu_emulate_irq(s->guest, guest_irq, 0);
	if (rc) {
		vmm_printf("%s: Emulate Guest=%s irq=%d level=0 failed\n",
			   __func__, s->guest->name, guest_irq);
	}

	/* Elevate the interrupt level.
	 * This will force interrupt triggering.
	 */
	rc = vmm_devemu_emulate_irq(s->guest, guest_irq, 1);
	if (rc) {
		vmm_printf("%s: Emulate Guest=%s irq=%d level=1 failed\n",
			   __func__, s->guest->name, guest_irq);
	}

done:
	return VMM_IRQ_HANDLED;
}

static int simple_emulator_reset(struct vmm_emudev *edev)
{
	u32 i;
	struct simple_state *s = edev->priv;

	for (i = 0; i < s->irq_count; i++) {
		vmm_devemu_map_host2guest_irq(s->guest,
					      s->guest_irqs[i],
					      s->host_irqs[i]);
	}

	return VMM_OK;
}

static int simple_emulator_probe(struct vmm_guest *guest,
				 struct vmm_emudev *edev,
				 const struct vmm_devtree_nodeid *eid)
{
	int rc = VMM_OK;
	u32 i, irq_reg_count = 0;
	struct simple_state *s;

	s = vmm_zalloc(sizeof(struct simple_state));
	if (!s) {
		rc = VMM_ENOMEM;
		goto simple_emulator_probe_fail;
	}

	strlcpy(s->name, guest->name, sizeof(s->name));
	strlcat(s->name, "/", sizeof(s->name));
	if (strlcat(s->name, edev->node->name, sizeof(s->name)) >=
							sizeof(s->name)) {
		rc = VMM_EOVERFLOW;
		goto simple_emulator_probe_freestate_fail;
	}

	s->guest = guest;
	s->irq_count = vmm_devtree_irq_count(edev->node);
	s->guest_irqs = NULL;
	s->host_irqs = NULL;

	i = vmm_devtree_attrlen(edev->node, "host-interrupts") / sizeof(u32);
	if (s->irq_count != i) {
		rc = VMM_EINVALID;
		goto simple_emulator_probe_freestate_fail;
	}

	if (s->irq_count) {
		s->host_irqs = vmm_zalloc(sizeof(u32) * s->irq_count);
		if (!s->host_irqs) {
			rc = VMM_ENOMEM;
			goto simple_emulator_probe_freestate_fail;
		}

		s->guest_irqs = vmm_zalloc(sizeof(u32) * s->irq_count);
		if (!s->guest_irqs) {
			rc = VMM_ENOMEM;
			goto simple_emulator_probe_freehirqs_fail;
		}
	}

	for (i = 0; i < s->irq_count; i++) {
		rc = vmm_devtree_read_u32_atindex(edev->node,
						  "host-interrupts",
						  &s->host_irqs[i], i);
		if (rc) {
			goto simple_emulator_probe_cleanupirqs_fail;
		}

		rc = vmm_devtree_irq_get(edev->node, &s->guest_irqs[i], i);
		if (rc) {
			goto simple_emulator_probe_cleanupirqs_fail;
		}

		rc = vmm_host_irq_mark_routed(s->host_irqs[i]);
		if (rc) {
			goto simple_emulator_probe_cleanupirqs_fail;
		}

		rc = vmm_host_irq_register(s->host_irqs[i], s->name,
					   simple_routed_irq, s);
		if (rc) {
			vmm_host_irq_unmark_routed(s->host_irqs[i]);
			goto simple_emulator_probe_cleanupirqs_fail;
		}

		irq_reg_count++;
	}

	edev->priv = s;

	return VMM_OK;

simple_emulator_probe_cleanupirqs_fail:
	for (i = 0; i < irq_reg_count; i++) {
		vmm_host_irq_unregister(s->host_irqs[i], s);
		vmm_host_irq_unmark_routed(s->host_irqs[i]);
	}
	if (s->guest_irqs) {
		vmm_free(s->guest_irqs);
	}
simple_emulator_probe_freehirqs_fail:
	if (s->host_irqs) {
		vmm_free(s->host_irqs);
	}
simple_emulator_probe_freestate_fail:
	vmm_free(s);
simple_emulator_probe_fail:
	return rc;
}

static int simple_emulator_remove(struct vmm_emudev *edev)
{
	u32 i;
	struct simple_state *s = edev->priv;

	if (!s) {
		return VMM_EFAIL;
	}

	for (i = 0; i < s->irq_count; i++) {
		vmm_host_irq_unregister(s->host_irqs[i], s);
		vmm_host_irq_unmark_routed(s->host_irqs[i]);
	}
	if (s->guest_irqs) {
		vmm_free(s->guest_irqs);
	}
	if (s->host_irqs) {
		vmm_free(s->host_irqs);
	}
	vmm_free(s);

	edev->priv = NULL;

	return VMM_OK;
}

static struct vmm_devtree_nodeid simple_emuid_table[] = {
	{ .type = "pt", .compatible = "simple", },
	{ /* end of list */ },
};

static struct vmm_emulator simple_emulator = {
	.name = "simple",
	.match_table = simple_emuid_table,
	.endian = VMM_DEVEMU_NATIVE_ENDIAN,
	.probe = simple_emulator_probe,
	.reset = simple_emulator_reset,
	.remove = simple_emulator_remove,
};

static int __init simple_emulator_init(void)
{
	return vmm_devemu_register_emulator(&simple_emulator);
}

static void __exit simple_emulator_exit(void)
{
	vmm_devemu_unregister_emulator(&simple_emulator);
}

VMM_DECLARE_MODULE(MODULE_DESC,
			MODULE_AUTHOR,
			MODULE_LICENSE,
			MODULE_IPRIORITY,
			MODULE_INIT,
			MODULE_EXIT);
