/**
 * Copyright (c) 2011 Anup Patel.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @file cpu_defines.h
 * @author Anup Patel (anup@brainfault.org)
 * @brief common macros & defines for shared by all C & Assembly code
 */
#ifndef __CPU_DEFINES_H__
#define __CPU_DEFINES_H__

/* Maximum allowed VTLB entries */
#define CPU_VCPU_VTLB_ZONE_V				0
#define CPU_VCPU_VTLB_ZONE_V_LEN			16
#define CPU_VCPU_VTLB_ZONE_HVEC				1
#define CPU_VCPU_VTLB_ZONE_HVEC_LEN			16
#define CPU_VCPU_VTLB_ZONE_LVEC				2
#define CPU_VCPU_VTLB_ZONE_LVEC_LEN			16
#define CPU_VCPU_VTLB_ZONE_G				3
#define CPU_VCPU_VTLB_ZONE_G_LEN			256
#define CPU_VCPU_VTLB_ZONE_NG				4
#define CPU_VCPU_VTLB_ZONE_NG_LEN			128
#define CPU_VCPU_VTLB_ZONE_COUNT			5
#define CPU_VCPU_VTLB_ENTRY_COUNT			\
				(CPU_VCPU_VTLB_ZONE_V_LEN + \
				 CPU_VCPU_VTLB_ZONE_HVEC_LEN + \
				 CPU_VCPU_VTLB_ZONE_LVEC_LEN + \
				 CPU_VCPU_VTLB_ZONE_G_LEN + \
				 CPU_VCPU_VTLB_ZONE_NG_LEN)

/* Coprocessor related macros & defines */
#define CPU_COPROC_COUNT				16

/* GPR related macros & defines */
#define CPU_GPR_COUNT 					13
#define CPU_FIQ_GPR_COUNT 				5

/* Interrupt or Exception related macros & defines */
#define CPU_IRQ_NR					8
#define CPU_IRQ_LOWVEC_BASE				0x00000000
#define CPU_IRQ_HIGHVEC_BASE				0xFFFF0000
#define CPU_RESET_IRQ					0
#define CPU_UNDEF_INST_IRQ				1
#define CPU_SOFT_IRQ					2
#define CPU_PREFETCH_ABORT_IRQ				3
#define CPU_DATA_ABORT_IRQ				4
#define CPU_NOT_USED_IRQ				5
#define CPU_EXTERNAL_IRQ				6
#define CPU_EXTERNAL_FIQ				7

/* CPSR related macros & defines */
#define CPSR_MODE_MASK					0x0000001f
#define CPSR_MODE_USER					0x00000010
#define CPSR_MODE_FIQ					0x00000011
#define CPSR_MODE_IRQ					0x00000012
#define CPSR_MODE_SUPERVISOR				0x00000013
#define CPSR_MODE_MONITOR				0x00000016
#define CPSR_MODE_ABORT					0x00000017
#define CPSR_MODE_UNDEFINED				0x0000001b
#define CPSR_MODE_SYSTEM				0x0000001f
#define CPSR_THUMB_ENABLED				(1 << 5)
#define CPSR_FIQ_DISABLED				(1 << 6)
#define CPSR_IRQ_DISABLED				(1 << 7)
#define CPSR_ASYNC_ABORT_DISABLED			(1 << 8)
#define CPSR_BE_ENABLED					(1 << 9)
#define CPSR_IT2_MASK					0x0000FC00
#define CPSR_IT2_SHIFT					10
#define CPSR_GE_MASK					0x000F0000
#define CPSR_GE_SHIFT					16
#define CPSR_JAZZLE_ENABLED				(1 << 24)
#define CPSR_IT1_MASK					0x06000000
#define CPSR_IT1_SHIFT					25
#define CPSR_CUMMULATE_MASK				(1 << 27)
#define CPSR_CUMMULATE_SHIFT				27
#define CPSR_OVERFLOW_MASK				(1 << 28)
#define CPSR_OVERFLOW_SHIFT				28
#define CPSR_CARRY_MASK					(1 << 29)
#define CPSR_CARRY_SHIFT				29
#define CPSR_ZERO_MASK					(1 << 30)
#define CPSR_ZERO_SHIFT					30
#define CPSR_NEGATIVE_MASK				(1 << 31)
#define CPSR_NEGATIVE_SHIFT				31

#define CPSR_NZCV_MASK					(CPSR_NEGATIVE_MASK |\
							CPSR_ZERO_MASK |\
							CPSR_CARRY_MASK |\
							CPSR_OVERFLOW_MASK)
#define CPSR_IT_MASK					(CPSR_IT2_MASK |\
							CPSR_IT1_MASK)
#define CPSR_USERBITS_MASK				(CPSR_NZCV_MASK |\
							CPSR_CUMMULATE_MASK |\
							CPSR_GE_MASK |\
							CPSR_IT_MASK |\
							CPSR_THUMB_ENABLED)
#define CPSR_PRIVBITS_MASK				(~CPSR_USERBITS_MASK)
#define CPSR_ALLBITS_MASK				0xFFFFFFFF

/* VFP system registers.  */
#define VFP_FPSID					0
#define VFP_FPSCR					1
#define VFP_MVFR1					6
#define VFP_MVFR0					7
#define VFP_FPEXC					8
#define VFP_FPINST					9
#define VFP_FPINST2					10

/* iwMMXt coprocessor control registers.  */
#define IWMMXT_wCID					0
#define IWMMXT_wCon					1
#define IWMMXT_wCSSF					2
#define IWMMXT_wCASF					3
#define IWMMXT_wCGR0					8
#define IWMMXT_wCGR1					9
#define IWMMXT_wCGR2					10
#define IWMMXT_wCGR3					11

/* SCTLR related macros & defines */
#define SCTLR_TE_MASK					0x40000000
#define SCTLR_AFE_MASK					0x20000000
#define SCTLR_TRE_MASK					0x10000000
#define SCTLR_NFI_MASK					0x08000000
#define SCTLR_EE_MASK					0x02000000
#define SCTLR_VE_MASK					0x01000000
#define SCTLR_V6_MASK					0x00800000
#define SCTLR_U_MASK					0x00400000
#define SCTLR_FI_MASK					0x00200000
#define SCTLR_HA_MASK					0x00020000
#define SCTLR_RR_MASK					0x00004000
#define SCTLR_L4_MASK					0x00004000
#define SCTLR_V_MASK					0x00002000
#define SCTLR_I_MASK					0x00001000
#define SCTLR_Z_MASK					0x00000800
#define SCTLR_SW_MASK					0x00000400
#define SCTLR_R_MASK					0x00000200
#define SCTLR_S_MASK					0x00000100
#define SCTLR_B_MASK					0x00000080
#define SCTLR_C_MASK					0x00000004
#define SCTLR_A_MASK					0x00000002
#define SCTLR_M_MASK					0x00000001
#define SCTLR_ROBITS_MASK				0x849D8378
#define SCTLR_V6_ROBITS_MASK				0xB01F0470
#define SCTLR_V6K_ROBITS_MASK				0x801D0470
#define SCTLR_V5_ROBITS_MASK				0xFFFF0C78
#define SCTLR_MMU_MASK					(SCTLR_M_MASK | \
							 SCTLR_TRE_MASK | \
							 SCTLR_AFE_MASK | \
							 SCTLR_U_MASK)

/* MPIDR related macros & defines */
#define MPIDR_SMP_BITMASK				(0x3 << 30)
#define MPIDR_SMP_VALUE					(0x2 << 30)
#define MPIDR_MT_BITMASK				(0x1 << 24)
#define MPIDR_HWID_BITMASK				0xFFFFFF
#define MPIDR_INVALID					(~MPIDR_HWID_BITMASK)
#define MPIDR_LEVEL_BITS_SHIFT				3
#define MPIDR_LEVEL_BITS				\
						(1 << MPIDR_LEVEL_BITS_SHIFT)
#define MPIDR_LEVEL_MASK				\
						((1 << MPIDR_LEVEL_BITS) - 1)
#define MPIDR_LEVEL_SHIFT(level)			\
			(((1 << level) >> 1) << MPIDR_LEVEL_BITS_SHIFT)
#define MPIDR_AFFINITY_LEVEL(mpidr, level)		\
		((mpidr >> (MPIDR_LEVEL_BITS * level)) & MPIDR_LEVEL_MASK)

/* CPACR related macros & define */
#define CPACR_ASEDIS_MASK				0x80000000
#define CPACR_ASEDIS_SHIFT				31
#define CPACR_D32DIS_MASK				0x40000000
#define CPACR_D32DIS_SHIFT				30
#define CPACR_TRCDIS_MASK				0x10000000
#define CPACR_TRCDIS_SHIFT				28
#define CPACR_CP_MASK(n)				(0x3 << ((n)*2))
#define CPACR_CP_SHIFT(n)				((n)*2)

/* CTR related macros & defines */
#define CTR_FORMAT_MASK					0xE0000000
#define CTR_FORMAT_SHIFT				29
#define CTR_FORMAT_V6					0x0
#define CTR_FORMAT_V7					0x4
#define CTR_CWG_MASK					0x0F000000
#define CTR_CWG_SHIFT					24
#define CTR_ERG_MASK					0x00F00000
#define CTR_ERG_SHIFT					20
#define CTR_DMINLINE_MASK				0x000F0000
#define CTR_DMINLINE_SHIFT				20
#define CTR_L1IP_MASK					0x0000C000
#define CTR_L1IP_SHIFT					14
#define CTR_IMINLINE_MASK				0x0000000F
#define CTR_IMINLINE_SHIFT				0
#define CTR_V6_CTYPE_MASK				0x1E000000
#define CTR_V6_CTYPE_SHIFT				25
#define CTR_V6_S_MASK					0x01000000
#define CTR_V6_S_SHIFT					24
#define CTR_V6_DSIZE_MASK				0x00FFF000
#define CTR_V6_DSIZE_SHIFT				12
#define CTR_V6_ISIZE_MASK				0x00000FFF
#define CTR_V6_ISIZE_SHIFT				0

/* CLIDR related macros & defines */
#define CLIDR_LOUU_MASK					0x38000000
#define CLIDR_LOUU_SHIFT				27
#define CLIDR_LOC_MASK					0x07000000
#define CLIDR_LOC_SHIFT					24
#define CLIDR_LOUIS_MASK				0x00E00000
#define CLIDR_LOUIS_SHIFT				21
#define CLIDR_CTYPE7_MASK				0x001C0000
#define CLIDR_CTYPE7_SHIFT				18
#define CLIDR_CTYPE6_MASK				0x00038000
#define CLIDR_CTYPE6_SHIFT				15
#define CLIDR_CTYPE5_MASK				0x00007000
#define CLIDR_CTYPE5_SHIFT				12
#define CLIDR_CTYPE4_MASK				0x00000E00
#define CLIDR_CTYPE4_SHIFT				9
#define CLIDR_CTYPE3_MASK				0x000001C0
#define CLIDR_CTYPE3_SHIFT				6
#define CLIDR_CTYPE2_MASK				0x00000038
#define CLIDR_CTYPE2_SHIFT				3
#define CLIDR_CTYPE1_MASK				0x00000007
#define CLIDR_CTYPE1_SHIFT				0
#define CLIDR_CTYPE_NOCACHE				0x0
#define CLIDR_CTYPE_ICACHE				0x1
#define CLIDR_CTYPE_DCACHE				0x2
#define CLIDR_CTYPE_SPLITCACHE				0x3
#define CLIDR_CTYPE_UNICACHE				0x4
#define CLIDR_CTYPE_RESERVED1				0x5
#define CLIDR_CTYPE_RESERVED2				0x6
#define CLIDR_CTYPE_RESERVED3				0x7

/* CSSELR related macros & defines */
#define CSSELR_LEVEL_MASK				0x0000000E
#define CSSELR_LEVEL_SHIFT				1
#define CSSELR_IND_MASK					0x00000001
#define CSSELR_IND_SHIFT				0

/* CSSIDR related macros & defines */
#define CCSIDR_WT_MASK					0x80000000
#define CCSIDR_WT_SHIFT					31
#define CCSIDR_WB_MASK					0x40000000
#define CCSIDR_WB_SHIFT					30
#define CCSIDR_RA_MASK					0x20000000
#define CCSIDR_RA_SHIFT					29
#define CCSIDR_WA_MASK					0x10000000
#define CCSIDR_WA_SHIFT					28
#define CCSIDR_NUMSETS_MASK				0x0FFFE000
#define CCSIDR_NUMSETS_SHIFT				13
#define CCSIDR_ASSOC_MASK				0x00001FF8
#define CCSIDR_ASSOC_SHIFT				3
#define CCSIDR_LINESZ_MASK				0x00000007
#define CCSIDR_LINESZ_SHIFT				0

/* Translation table related macros & defines */
#define TTBL_MIN_SIZE					0x1000
#define TTBL_MIN_PAGE_SIZE				TTBL_MIN_SIZE
#define TTBL_MIN_PAGE_MASK				(TTBL_MIN_PAGE_SIZE - 1)
#define TTBL_MIN_PAGE_SHIFT				12
#define TTBL_MAX_SIZE					0x4000
#define TTBL_MAX_PAGE_SIZE				0x1000000
#define TTBL_MAX_PAGE_MASK				(TTBL_MAX_PAGE_SIZE - 1)
#define TTBL_MAX_PAGE_SHIFT				24
#define TTBL_AP_S_U					0x0
#define TTBL_AP_SRW_U					0x1
#define TTBL_AP_SRW_UR					0x2
#define TTBL_AP_SRW_URW					0x3
#define TTBL_AP_SR_U					0x5
#define TTBL_AP_SR_UR_DEPRECATED			0x6
#define TTBL_AP_SR_UR					0x7
#define TTBL_DOM_MANAGER				0x3
#define TTBL_DOM_RESERVED				0x2
#define TTBL_DOM_CLIENT					0x1
#define TTBL_DOM_NOACCESS				0x0
#define TTBL_L1TBL_SIZE					0x4000
#define TTBL_L1TBL_MASK					(TTBL_L1TBL_SIZE - 1)
#define TTBL_L1TBL_SIZE_SHIFT				14
#define TTBL_L1TBL_SECTION_PAGE_SHIFT			20
#define TTBL_L1TBL_SECTION_PAGE_SIZE			(1 << TTBL_L1TBL_SECTION_PAGE_SHIFT)
#define TTBL_L1TBL_SECTION_PAGE_MASK			(TTBL_L1TBL_SECTION_PAGE_SIZE - 1)
#define TTBL_L1TBL_SUPSECTION_PAGE_SIZE			0x1000000
#define TTBL_L1TBL_SUPSECTION_PAGE_MASK			(TTBL_L1TBL_SUPSECTION_PAGE_SIZE - 1)
#define TTBL_L1TBL_TTE_OFFSET_MASK			0xFFF00000
#define TTBL_L1TBL_TTE_OFFSET_SHIFT			20
#define TTBL_L1TBL_TTE_BASE24_MASK			0xFF000000
#define TTBL_L1TBL_TTE_BASE24_SHIFT			24
#define TTBL_L1TBL_TTE_BASE20_MASK			0xFFF00000
#define TTBL_L1TBL_TTE_BASE20_SHIFT			20
#define TTBL_L1TBL_TTE_BASE10_MASK			0xFFFFFC00
#define TTBL_L1TBL_TTE_BASE10_SHIFT			10
#define TTBL_L1TBL_TTE_NS2_MASK				0x00080000
#define TTBL_L1TBL_TTE_NS2_SHIFT			19
#define TTBL_L1TBL_TTE_SECTYPE_MASK			0x00040000
#define TTBL_L1TBL_TTE_SECTYPE_SHIFT			18
#define TTBL_L1TBL_TTE_NG_MASK				0x00020000
#define TTBL_L1TBL_TTE_NG_SHIFT				17
#define TTBL_L1TBL_TTE_S_MASK				0x00010000
#define TTBL_L1TBL_TTE_S_SHIFT				16
#define TTBL_L1TBL_TTE_AP2_MASK				0x00008000
#define TTBL_L1TBL_TTE_AP2_SHIFT			15
#define TTBL_L1TBL_TTE_TEX_MASK				0x00007000
#define TTBL_L1TBL_TTE_TEX_SHIFT			12
#define TTBL_L1TBL_TTE_AP_MASK				0x00000C00
#define TTBL_L1TBL_TTE_AP_SHIFT				10
#define TTBL_L1TBL_TTE_IMP_MASK				0x00000200
#define TTBL_L1TBL_TTE_IMP_SHIFT			9
#define TTBL_L1TBL_TTE_DOM_MASK				0x000001E0
#define TTBL_L1TBL_TTE_DOM_SHIFT			5
#define TTBL_L1TBL_TTE_DOM_RESERVED			0x0
#define TTBL_L1TBL_TTE_DOM_VCPU_SUPER			0x1
#define TTBL_L1TBL_TTE_DOM_VCPU_SUPER_RW_USER_R		0x2
#define TTBL_L1TBL_TTE_DOM_VCPU_USER			0x3
#define TTBL_L1TBL_TTE_XN_SHIFT				4
#define TTBL_L1TBL_TTE_XN_MASK				(1 << TTBL_L1TBL_TTE_XN_SHIFT)
#define TTBL_L1TBL_TTE_REQ_SHIFT			4
#define TTBL_L1TBL_TTE_REQ_MASK				(1 << TTBL_L1TBL_TTE_REQ_SHIFT)
#define TTBL_L1TBL_TTE_NS1_SHIFT			3
#define TTBL_L1TBL_TTE_NS1_MASK				(1 << TTBL_L1TBL_TTE_NS1_SHIFT)
#define TTBL_L1TBL_TTE_C_SHIFT				3
#define TTBL_L1TBL_TTE_C_MASK				(1 << TTBL_L1TBL_TTE_C_SHIFT)
#define TTBL_L1TBL_TTE_B_SHIFT				2
#define TTBL_L1TBL_TTE_B_MASK				(1 << TTBL_L1TBL_TTE_B_SHIFT)
#define TTBL_L1TBL_TTE_TYPE_MASK			0x00000003
#define TTBL_L1TBL_TTE_TYPE_SHIFT			0
#define TTBL_L1TBL_TTE_TYPE_FAULT			0x0
#define TTBL_L1TBL_TTE_TYPE_L2TBL			0x1
#define TTBL_L1TBL_TTE_TYPE_COARSE_L2TBL		TTBL_L1TBL_TTE_TYPE_L2TBL
#define TTBL_L1TBL_TTE_TYPE_SECTION			0x2
#define TTBL_L1TBL_TTE_TYPE_RESERVED			0x3
#define TTBL_L1TBL_TTE_TYPE_FINE_L2TBL			TTBL_L1TBL_TTE_TYPE_RESERVED
#define TTBL_L2TBL_SIZE					0x400
#define TTBL_L2TBL_SIZE_SHIFT				10
#define TTBL_L2TBL_LARGE_PAGE_SIZE			0x10000
#define TTBL_L2TBL_SMALL_PAGE_SIZE			0x1000
#define TTBL_L2TBL_TINY_PAGE_SIZE			0x400
#define TTBL_L2TBL_TTE_OFFSET_MASK			0x000FF000
#define TTBL_L2TBL_TTE_OFFSET_SHIFT			12
#define TTBL_L2TBL_TTE_BASE16_MASK			0xFFFF0000
#define TTBL_L2TBL_TTE_BASE16_SHIFT			16
#define TTBL_L2TBL_TTE_LXN_MASK				0x00008000
#define TTBL_L2TBL_TTE_LXN_SHIFT			15
#define TTBL_L2TBL_TTE_BASE12_MASK			0xFFFFF000
#define TTBL_L2TBL_TTE_BASE12_SHIFT			12
#define TTBL_L2TBL_TTE_LTEX_MASK			0x00007000
#define TTBL_L2TBL_TTE_LTEX_SHIFT			12
#define TTBL_L2TBL_TTE_NG_MASK				0x00000800
#define TTBL_L2TBL_TTE_NG_SHIFT				11
#define TTBL_L2TBL_TTE_S_MASK				0x00000400
#define TTBL_L2TBL_TTE_S_SHIFT				10
#define TTBL_L2TBL_TTE_AP2_MASK				0x00000200
#define TTBL_L2TBL_TTE_AP2_SHIFT			9
#define TTBL_L2TBL_TTE_STEX_MASK			0x000001C0
#define TTBL_L2TBL_TTE_STEX_SHIFT			6
#define TTBL_L2TBL_TTE_AP_SHIFT				4
#define TTBL_L2TBL_TTE_AP_MASK				(3 << TTBL_L2TBL_TTE_AP_SHIFT)
#define TTBL_L2TBL_TTE_V5_AP0_SHIFT			4
#define TTBL_L2TBL_TTE_V5_AP0_MASK			(3 << TTBL_L2TBL_TTE_V5_AP0_SHIFT)
#define TTBL_L2TBL_TTE_V5_AP1_SHIFT			6
#define TTBL_L2TBL_TTE_V5_AP1_MASK			(3 << TTBL_L2TBL_TTE_V5_AP1_SHIFT)
#define TTBL_L2TBL_TTE_V5_AP2_SHIFT			8
#define TTBL_L2TBL_TTE_V5_AP2_MASK			(3 << TTBL_L2TBL_TTE_V5_AP2_SHIFT)
#define TTBL_L2TBL_TTE_V5_AP3_SHIFT			10
#define TTBL_L2TBL_TTE_V5_AP3_MASK			(3 << TTBL_L2TBL_TTE_V5_AP3_SHIFT)
#define TTBL_L2TBL_TTE_C_MASK				0x00000008
#define TTBL_L2TBL_TTE_C_SHIFT				3
#define TTBL_L2TBL_TTE_B_MASK				0x00000004
#define TTBL_L2TBL_TTE_B_SHIFT				2
#define TTBL_L2TBL_TTE_SXN_MASK				0x00000001
#define TTBL_L2TBL_TTE_SXN_SHIFT			0
#define TTBL_L2TBL_TTE_TYPE_MASK			0x00000003
#define TTBL_L2TBL_TTE_TYPE_SHIFT			0
#define TTBL_L2TBL_TTE_TYPE_FAULT			0x0
#define TTBL_L2TBL_TTE_TYPE_LARGE			0x1
#define TTBL_L2TBL_TTE_TYPE_SMALL_X			0x2
#define TTBL_L2TBL_TTE_TYPE_SMALL			0x2
#define TTBL_L2TBL_TTE_TYPE_SMALL_XN			0x3
#define TTBL_L2TBL_TTE_TYPE_TINY			0x3
#define TTBL_INITIAL_L2TBL_COUNT			8
#define TTBL_INITIAL_L2TBL_SIZE				(TTBL_INITIAL_L2TBL_COUNT * TTBL_L2TBL_SIZE)

/* TTBR0 related macros & defines */
#define TTBR0_IGRN0_MASK				0x00000040
#define TTBR0_IRGN0_SHIFT				6
#define TTBR0_NOS_MASK					0x00000020
#define TTBR0_RGN_MASK					0x00000018
#define TTBR0_RGN_SHIFT					3
#define TTBR0_IMP_MASK					0x00000004
#define TTBR0_S_MASK					0x00000002
#define TTBR0_C_MASK					0x00000001
#define TTBR0_IGRN1_MASK				0x00000001

/* TTBR1 related macros & defines */
#define TTBR1_IGRN0_MASK				0x00000040
#define TTBR1_IRGN0_SHIFT				6
#define TTBR1_NOS_MASK					0x00000020
#define TTBR1_RGN_MASK					0x00000018
#define TTBR1_RGN_SHIFT					3
#define TTBR1_IMP_MASK					0x00000004
#define TTBR1_S_MASK					0x00000002
#define TTBR1_C_MASK					0x00000001
#define TTBR1_IGRN1_MASK				0x00000001

/* TTBCR related macros & defines */
#define TTBCR_PD1_MASK					0x00000020
#define TTBCR_PD2_MASK					0x00000010
#define TTBCR_N_MASK					0x00000007

/* IFSR related macros & defines */
#define IFSR_EXT_MASK					0x00001000
#define IFSR_EXT_SHIFT					12
#define IFSR_FS4_MASK					0x00000400
#define IFSR_FS4_SHIFT					10
#define IFSR_FS_MASK					0x0000000F
#define IFSR_FS_SHIFT					0
#define IFSR_FS_TTBL_WALK_SYNC_EXT_ABORT_1		12
#define IFSR_FS_TTBL_WALK_SYNC_EXT_ABORT_2		14
#define IFSR_FS_TTBL_WALK_SYNC_PARITY_ERROR_1		28
#define IFSR_FS_TTBL_WALK_SYNC_PARITY_ERROR_2		30
#define IFSR_FS_TRANS_FAULT_SECTION			5
#define IFSR_FS_TRANS_FAULT_PAGE			7
#define IFSR_FS_ACCESS_FAULT_SECTION			3
#define IFSR_FS_ACCESS_FAULT_PAGE			6
#define IFSR_FS_DOMAIN_FAULT_SECTION			9
#define IFSR_FS_DOMAIN_FAULT_PAGE			11
#define IFSR_FS_PERM_FAULT_SECTION			13
#define IFSR_FS_PERM_FAULT_PAGE				15
#define IFSR_FS_DEBUG_EVENT				2
#define IFSR_FS_SYNC_EXT_ABORT				8
#define IFSR_FS_IMP_VALID_LOCKDOWN			20
#define IFSR_FS_IMP_VALID_COPROC_ABORT			26
#define IFSR_FS_MEM_ACCESS_SYNC_PARITY_ERROR		25

/* DFSR related macros & defines */
#define DFSR_EXT_MASK					0x00001000
#define DFSR_EXT_SHIFT					12
#define DFSR_WNR_MASK					0x00000800
#define DFSR_WNR_SHIFT					11
#define DFSR_FS4_MASK					0x00000400
#define DFSR_FS4_SHIFT					10
#define DFSR_DOM_MASK					0x000000F0
#define DFSR_DOM_SHIFT					4
#define DFSR_FS_MASK					0x0000000F
#define DFSR_FS_SHIFT					0
#define DFSR_FS_ALIGN_FAULT				1
#define DFSR_FS_ICACHE_MAINT_FAULT			4
#define DFSR_FS_TTBL_WALK_SYNC_EXT_ABORT_1		12
#define DFSR_FS_TTBL_WALK_SYNC_EXT_ABORT_2		14
#define DFSR_FS_TTBL_WALK_SYNC_PARITY_ERROR_1		28
#define DFSR_FS_TTBL_WALK_SYNC_PARITY_ERROR_2		30
#define DFSR_FS_TRANS_FAULT_SECTION			5
#define DFSR_FS_TRANS_FAULT_PAGE			7
#define DFSR_FS_ACCESS_FAULT_SECTION			3
#define DFSR_FS_ACCESS_FAULT_PAGE			6
#define DFSR_FS_DOMAIN_FAULT_SECTION			9
#define DFSR_FS_DOMAIN_FAULT_PAGE			11
#define DFSR_FS_PERM_FAULT_SECTION			13
#define DFSR_FS_PERM_FAULT_PAGE				15
#define DFSR_FS_DEBUG_EVENT				2
#define DFSR_FS_SYNC_EXT_ABORT				8
#define DFSR_FS_IMP_VALID_LOCKDOWN			20
#define DFSR_FS_IMP_VALID_COPROC_ABORT			26
#define DFSR_FS_MEM_ACCESS_SYNC_PARITY_ERROR		25
#define DFSR_FS_ASYNC_EXT_ABORT				22
#define DFSR_FS_MEM_ACCESS_ASYNC_PARITY_ERROR		24

/* PAR */
#define PAR_PA_MASK					0xFFFFF000
#define PAR_PA_SHIFT					12
#define PAR_LPAE_MASK					0x00000800
#define PAR_LPAE_SHIFT					11
#define PAR_NOS_MASK					0x00000400
#define PAR_NOS_SHIFT					10
#define PAR_NS_MASK					0x00000200
#define PAR_NS_SHIFT					9
#define PAR_SH_MASK					0x00000080
#define PAR_SH_SHIFT					7
#define PAR_INNER_MASK					0x00000070
#define PAR_INNER_SHIFT					4
#define PAR_OUTER_MASK					0x0000000C
#define PAR_OUTEr_SHIFT					2
#define PAR_SS_MASK					0x00000002
#define PAR_SS_SHIFT					1
#define PAR_F_MASK					0x00000001
#define PAR_F_SHIFT					0

/* PAR64 */
#define PAR64_ATTR_MASK					0xFF00000000000000ULL
#define PAR64_ATTR_SHIFT				56
#define PAR64_PA_MASK					0x000000FFFFFFF000ULL
#define PAR64_PA_SHIFT					12
#define PAR64_LPAE_MASK					0x0000000000000800ULL
#define PAR64_LPAE_SHIFT				11
#define PAR64_NS_MASK					0x0000000000000200ULL
#define PAR64_NS_SHIFT					9
#define PAR64_SH_MASK					0x0000000000000180ULL
#define PAR64_SH_SHIFT					7
#define PAR64_F_MASK					0x0000000000000001ULL
#define PAR64_F_SHIFT					0

/* MIDR */
#define MIDR_IMPLEMENTER_MASK				0xFF000000
#define MIDR_IMPLEMENTER_SHIFT				24
#define MIDR_VARIANT_MASK				0x00F00000
#define MIDR_VARIANT_SHIFT				20
#define MIDR_ARCHITECTURE_MASK				0x000F0000
#define MIDR_ARCHITECTURE_SHIFT				16
#define MIDR_PARTNUM_MASK				0x0000FFF0
#define MIDR_PARTNUM_SHIFT				4
#define MIDR_REVISON_MASK				0x0000000F
#define MIDR_REVISON_SHIFT				0

/* FPEXC */
#define FPEXC_EX_MASK					(1u << 31)
#define FPEXC_EX_SHIFT					31
#define FPEXC_EN_MASK					(1u << 30)
#define FPEXC_EN_SHIFT					30
#define FPEXC_FP2V_MASK					(1u << 28)
#define FPEXC_FP2V_SHIFT				28

/* FPSID */
#define FPSID_IMPLEMENTER_MASK				(0xff << 24)
#define FPSID_IMPLEMENTER_SHIFT				(24)
#define FPSID_SW_MASK					(0x1 << 23)
#define FPSID_SW_SHIFT					(23)
#define FPSID_ARCH_MASK					(0x7f << 16)
#define FPSID_ARCH_SHIFT				(16)
#define FPSID_PART_MASK					(0xff << 8)
#define FPSID_PART_SHIFT				(8)
#define FPSID_VARIANT_MASK				(0xf << 4)
#define FPSID_VARIANT_SHIFT				(4)
#define FPSID_REV_MASK					(0xf << 0)
#define FPSID_REV_SHIFT					(0)

/* MVFR0 */
#define MVFR0_VFP_ROUND_MODES_MASK			(0xf << 28)
#define MVFR0_VFP_ROUND_MODES_SHIFT			28
#define MVFR0_SHORT_VECTORS_MASK			(0xf << 24)
#define MVFR0_SHORT_VECTORS_SHIFT			24
#define MVFR0_SQUARE_ROOT_MASK				(0xf << 20)
#define MVFR0_SQUARE_ROOT_SHIFT				20
#define MVFR0_DIVIDE_MASK				(0xf << 16)
#define MVFR0_DIVIDE_SHIFT				16
#define MVFR0_VFP_EXEC_TRAP_MASK			(0xf << 12)
#define MVFR0_VFP_EXEC_TRAP_SHIFT			12
#define MVFR0_DOUBLE_PREC_MASK				(0xf << 8)
#define MVFR0_DOUBLE_PREC_SHIFT				8
#define MVFR0_SINGLE_PREC_MASK				(0xf << 4)
#define MVFR0_SINGLE_PREC_SHIFT				4
#define MVFR0_A_SIMD_MASK				(0xf << 0)
#define MVFR0_A_SIMD_SHIFT				0

/* ID_PFR0 */
#define ID_PFR0_STATE3_MASK				0x0000f000
#define ID_PFR0_STATE3_SHIFT				12
#define ID_PFR0_STATE2_MASK				0x00000f00
#define ID_PFR0_STATE2_SHIFT				8
#define ID_PFR0_STATE1_MASK				0x000000f0
#define ID_PFR0_STATE1_SHIFT				4
#define ID_PFR0_STATE0_MASK				0x00000000
#define ID_PFR0_STATE0_SHIFT				0

/* ID_PFR1 */
#define ID_PFR1_GEN_TIMER_MASK				0x000f0000
#define ID_PFR1_GEN_TIMER_SHIFT				16
#define ID_PFR1_VIRTEX_MASK				0x0000f000
#define ID_PFR1_VIRTEX_SHIFT				12
#define ID_PFR1_M_PROFILE_MASK				0x00000f00
#define ID_PFR1_M_PROFILE_SHIFT				8
#define ID_PFR1_SECUREX_MASK				0x000000f0
#define ID_PFR1_SECUREX_SHIFT				4
#define ID_PFR1_PRG_MODEL_MASK				0x0000000f
#define ID_PFR1_PRG_MODEL_SHIFT				0

#endif
